# Staff Login & Dashboard Guide

## Overview
A complete staff management system with dedicated login and responsive dashboard for cleaning service staff members.

### Features
- ✅ **Staff Login** - Secure staff member authentication
- ✅ **Responsive Dashboard** - Works on all devices (desktop, tablet, mobile)
- ✅ **Assignment Tracking** - View upcoming and all assignments
- ✅ **Assignment Status** - Track pending, confirmed, and completed work
- ✅ **Profile Management** - View personal profile and assigned services
- ✅ **Real-time Statistics** - Dashboard statistics update automatically

---

## Staff Login (staff-login.php)

### Access
- URL: `http://localhost/Cleaning%20Service/staff-login.php`
- Redirects automatically if already logged in

### Login Features
1. **Email/Password Authentication**
   - Enter registered staff email
   - Enter password
   - Password verification with database

2. **Remember Me**
   - Check "Remember me" to stay logged in for 30 days
   - Creates secure cookies

3. **Responsive Design**
   - Desktop: Full-width form
   - Tablet: Optimized spacing
   - Mobile: Touch-friendly interface

### Security Features
- Session management
- Password verification
- Active status checking
- Automatic redirects based on login state

### Error Handling
- Invalid credentials message
- Inactive account notification
- Server connection errors
- Helpful navigation links

---

## Staff Dashboard (staff-dashboard.php)

### Access
- URL: `http://localhost/Cleaning%20Service/staff-dashboard.php`
- Requires staff login
- Session-based access control

### Dashboard Sections

#### 1. Dashboard Tab (Default)

**Statistics Cards:**
- Total Assignments
- Pending Assignments
- Confirmed Assignments
- Completed Assignments

**Upcoming Assignments Table:**
- Shows next assignments to be completed
- Columns: Customer, Service, Date, Time, Status
- Limited to 5 most recent

#### 2. My Assignments Tab

**Filter Options:**
- All - Show all assignments
- Pending - Awaiting confirmation
- Confirmed - Ready to be completed
- Completed - Already finished

**Assignment Details:**
- Customer name
- Service being provided
- Booking date and time
- Location/address
- Current status
- Action buttons (Mark Complete)

#### 3. My Profile Tab

**Profile Information:**
- Full name
- Position/Role
- Email address
- Phone number
- Assigned services
- Account status

---

## Responsive Design Breakdown

### Desktop (769px and above)
- Two-column layout (sidebar + content)
- Full sidebar visible
- Full data tables
- All information visible at once

### Tablet (600px - 768px)
- Collapsible sidebar (click hamburger menu)
- Optimized spacing
- Readable tables
- Touch-friendly buttons

### Mobile (Below 600px)
- Full-width layout
- Hamburger menu for navigation
- Stacked information cards
- Full-width buttons
- Optimized font sizes
- Single-column tables with horizontal scroll

---

## API Integration

### Staff Login Endpoint

**Endpoint:** `POST api_handler.php`

**Parameters:**
```
action: staff_login
email: staff_email@example.com
password: staff_password
remember_me: 1 or 0
```

**Response Success:**
```json
{
  "success": true,
  "message": "Login successful"
}
```

**Response Error:**
```json
{
  "success": false,
  "message": "Invalid password"
}
```

### Dashboard Endpoints

#### Get Staff Stats
```
GET api_handler.php?action=get_staff_stats
```

**Response:**
```json
{
  "success": true,
  "data": {
    "total_assignments": 12,
    "pending_assignments": 3,
    "confirmed_assignments": 5,
    "completed_assignments": 4
  }
}
```

#### Get Staff Profile
```
GET api_handler.php?action=get_staff_profile
```

**Response:**
```json
{
  "success": true,
  "data": {
    "id": 1,
    "name": "John Doe",
    "email": "john@example.com",
    "phone": "1234567890",
    "position": "Cleaner",
    "status": "active",
    "services": [
      {"id": 1, "name": "Room Cleaning"},
      {"id": 2, "name": "Kitchen Cleaning"}
    ]
  }
}
```

#### Get Upcoming Assignments
```
GET api_handler.php?action=get_staff_upcoming_assignments
```

**Response:**
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "customer_name": "John Smith",
      "service_name": "Room Cleaning",
      "booking_date": "2026-02-04",
      "booking_time": "10:00",
      "status": "confirmed"
    }
  ]
}
```

#### Get All Assignments
```
GET api_handler.php?action=get_staff_assignments&status=all|pending|confirmed|completed
```

**Response:**
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "customer_name": "John Smith",
      "service_name": "Room Cleaning",
      "booking_date": "2026-02-04",
      "booking_time": "10:00",
      "address": "123 Main St",
      "status": "confirmed"
    }
  ]
}
```

#### Complete Assignment
```
POST api_handler.php
action: complete_staff_assignment
assignment_id: 1
```

**Response:**
```json
{
  "success": true,
  "message": "Assignment marked as complete"
}
```

---

## Database Schema

### Staff Table
```sql
CREATE TABLE staff (
  id INT PRIMARY KEY AUTO_INCREMENT,
  name VARCHAR(255) NOT NULL,
  email VARCHAR(255) UNIQUE NOT NULL,
  phone VARCHAR(20) NOT NULL,
  position VARCHAR(100) NOT NULL,
  password VARCHAR(255) NOT NULL,
  status ENUM('active', 'inactive') DEFAULT 'active',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)
```

### Staff Services Table (Staff-Service Mapping)
```sql
CREATE TABLE staff_services (
  id INT PRIMARY KEY AUTO_INCREMENT,
  staff_id INT NOT NULL,
  service_id INT NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (staff_id) REFERENCES staff(id) ON DELETE CASCADE,
  FOREIGN KEY (service_id) REFERENCES services(id) ON DELETE CASCADE,
  UNIQUE KEY unique_staff_service (staff_id, service_id)
)
```

### Staff Bookings Table (Staff-Booking Assignment)
```sql
CREATE TABLE staff_bookings (
  id INT PRIMARY KEY AUTO_INCREMENT,
  staff_id INT NOT NULL,
  booking_id INT NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (staff_id) REFERENCES staff(id) ON DELETE CASCADE,
  FOREIGN KEY (booking_id) REFERENCES bookings(id) ON DELETE CASCADE,
  UNIQUE KEY unique_staff_booking (staff_id, booking_id)
)
```

---

## Setup Instructions

### 1. Database Setup
1. Navigate to `http://localhost/Cleaning%20Service/setup_database.php`
2. The script will create all necessary tables
3. Verify in the output message

### 2. Create Staff Members
1. Log in to admin dashboard
2. Go to "Staff Management"
3. Click "Add Staff Member"
4. Fill in details:
   - Name
   - Email (unique)
   - Phone
   - Position
   - Select Services
   - Set Status (Active/Inactive)

### 3. Assign Staff to Bookings
1. When a booking is created/edited in admin
2. Assign staff members to the booking
3. Staff will see the assignment in their dashboard

### 4. Staff Login
1. Staff navigate to: `staff-login.php`
2. Enter email and password
3. Check "Remember me" (optional)
4. Click "Login"
5. Access the staff dashboard

---

## Features & Functionality

### Real-time Updates
- Statistics update when viewing dashboard
- Assignments load fresh when switching tabs
- Status changes reflect immediately

### Navigation
- Sidebar menu with three main sections
- Mobile hamburger menu
- Active tab highlighting
- Automatic page title updates

### Assignment Management
- Filter by status (All, Pending, Confirmed, Completed)
- Click "Mark Complete" to finish assignment
- Confirmation dialog before marking complete

### Profile Management
- View complete profile information
- See assigned services
- Check account status
- Edit profile (future enhancement)

---

## Security Considerations

1. **Authentication**
   - Session-based login system
   - Password verification
   - Automatic logout (server-side session timeout)

2. **Authorization**
   - Only staff can access staff dashboard
   - Admin-only staff management
   - User type validation

3. **Data Protection**
   - SQL injection prevention via parameterized queries
   - Input validation and sanitization
   - Secure password handling

---

## Responsive Behavior

### Sidebar
- **Desktop**: Always visible
- **Tablet/Mobile**: Toggle-able with hamburger menu
- **Auto-hide**: Closes when navigation link clicked on mobile

### Content
- **Flexbox Layout**: Adapts to screen size
- **Grid System**: Responsive cards and tables
- **Touch Optimization**: Large tap targets on mobile

### Tables
- **Desktop**: Full display
- **Mobile**: Horizontal scroll with truncated text

---

## Troubleshooting

### Staff Login Not Working
1. Check email and password are correct
2. Verify staff is set to "active" status
3. Ensure staff record exists in database
4. Check error message for details

### Dashboard Not Loading
1. Verify staff is logged in (check session)
2. Check browser console for errors
3. Verify API endpoints are accessible
4. Check database connection

### Assignments Not Showing
1. Verify bookings exist in database
2. Check staff_bookings table for assignments
3. Verify booking date is current or future
4. Check booking status is not cancelled

### Responsive Issues
1. Clear browser cache
2. Check viewport meta tag
3. Test in different browsers
4. Verify CSS media queries loading

---

## Mobile Testing Checklist

- [ ] Login form is fully visible
- [ ] Buttons are clickable without zooming
- [ ] Text is readable (16px+)
- [ ] No horizontal scroll needed
- [ ] Hamburger menu works
- [ ] Tab navigation works
- [ ] Forms are usable on mobile
- [ ] Tables are scrollable
- [ ] Colors contrast is good
- [ ] All features work

---

## Files Modified/Created

### New Files
- `staff-login.php` - Staff login page
- `staff-dashboard.php` - Staff dashboard
- `STAFF_DASHBOARD_GUIDE.md` - This documentation

### Updated Files
- `api_handler.php` - Added staff login and dashboard endpoints
- `setup_database.php` - Added staff and staff_bookings tables
- `admin-dashboard.js` - Fixed profile loading issue

---

## Performance Notes

- Statistics calculated on-demand (not stored)
- Assignments filtered server-side
- Pagination recommended for large datasets
- Caching can be added in future versions

---

## Future Enhancements

1. Staff profile editing
2. Real-time notifications
3. Assignment scheduling
4. Customer ratings by assignment
5. Performance metrics dashboard
6. Availability/scheduling system
7. Two-factor authentication
8. Staff mobile app

---

## Support

For issues or questions:
1. Check this documentation first
2. Review database tables
3. Check API responses in browser dev tools
4. Review server error logs
5. Verify all files are in place

